# scripts/build_kernel.py
#!/usr/bin/env python3
"""
Construct the reproduction kernel matrix and save it as a NumPy .npy file.
Reads context levels (n) from D_values.csv and pivot parameters from pivot_params.json.
"""
import argparse
import json
import numpy as np
from ar_sim.common.kernel_builder import build_reproduction_kernel


def main():
    parser = argparse.ArgumentParser(
        description="Build reproduction kernel from fractal anchors and pivot params"
    )
    parser.add_argument(
        "--n-values", required=False,
        help="Path to CSV file with 'n' column, default=data/D_values.csv",
        default="data/D_values.csv"
    )
    parser.add_argument(
        "--pivot-params", required=False,
        help="Path to JSON file with pivot parameters a,b,D_vals, default=data/pivot_params.json",
        default="data/pivot_params.json"
    )
    parser.add_argument(
        "--sigma", type=float, default=1.0,
        help="Kernel sigma (width) parameter"
    )
    parser.add_argument(
        "--out", required=False,
        help="Output .npy file path for kernel matrix",
        default="data/kernel.npy"
    )
    args = parser.parse_args()

    # Load context levels
    data = np.loadtxt(args.n_values, delimiter=',', skiprows=1, usecols=0)
    n_vals = data

    # Load pivot params
    with open(args.pivot_params, 'r') as f:
        pivot = json.load(f)
    # Ensure D_vals provided
    if 'D_vals' not in pivot:
        # load D column
        D_vals = np.loadtxt(args.n_values, delimiter=',', skiprows=1, usecols=1)
        pivot['D_vals'] = D_vals.tolist()

    # Build kernel
    M = build_reproduction_kernel(n_vals, pivot, sigma=args.sigma)
    # Save
    np.save(args.out, M)
    print(f"Saved kernel matrix to {args.out}")

if __name__ == '__main__':
    main()
